/*
 *  Copyright 2016, Plutext Pty Ltd.
 *   
 *  This file is part of docx4j.

    docx4j is licensed under the Apache License, Version 2.0 (the "License"); 
    you may not use this file except in compliance with the License. 

    You may obtain a copy of the License at 

        http://www.apache.org/licenses/LICENSE-2.0 

    Unless required by applicable law or agreed to in writing, software 
    distributed under the License is distributed on an "AS IS" BASIS, 
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
    See the License for the specific language governing permissions and 
    limitations under the License.

 */

package org.pptx4j.samples;


import java.util.ArrayList;
import java.util.List;

import org.docx4j.XmlUtils;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.openpackaging.packages.PresentationMLPackage;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.PresentationML.MainPresentationPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideLayoutPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideMasterPart;
import org.docx4j.openpackaging.parts.PresentationML.SlidePart;
import org.pptx4j.jaxb.Context;
import org.pptx4j.pml.Shape;
import org.pptx4j.pml.Sld;
import org.pptx4j.pml.SldLayout;
import org.pptx4j.pml.SldMaster;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



/**
 * Example of adding a slide, and populating it from the corresponding
 * layout, so its placeholders are visible in Powerpoint.
 * 
 * @author jharrop
 *
 */
public class SlidePlaceholder  {
	
	protected static Logger log = LoggerFactory.getLogger(SlidePlaceholder.class);
			
	public static void main(String[] args) throws Exception {

		// Where will we save our new .ppxt?
		String outputfilepath = System.getProperty("user.dir") + "/OUT_SlidePlaceholder.pptx";
		
		// Create skeletal package, including a MainPresentationPart and a SlideLayoutPart
		PresentationMLPackage presentationMLPackage = PresentationMLPackage.createPackage(); 
		
		// Need references to these parts to create a slide
		// Please note that these parts *already exist* - they are
		// created by createPackage() above.  See that method
		// for instruction on how to create and add a part.
		MainPresentationPart pp = presentationMLPackage.getMainPresentationPart();
		
		SlideLayoutPart layoutPart = (SlideLayoutPart)presentationMLPackage.getParts().getParts().get(
				new PartName("/ppt/slideLayouts/slideLayout1.xml"));
		layoutPart.setContents(
				(SldLayout)XmlUtils.unmarshalString(sldLayoutXML, Context.jcPML));
		
		
		SlideMasterPart masterPart = layoutPart.getSlideMasterPart();
		masterPart.setContents(
					(SldMaster)XmlUtils.unmarshalString(slideMasterXML, Context.jcPML));
		/* NB, master contains something like:
		 * 
			  <p:sldLayoutIdLst>
			    <p:sldLayoutId id="2147483649" r:id="rId1"/>
			    :
			  </p:sldLayoutIdLst>
			  
			  Powerpoint will be unable to open the pptx if any of these are missing.
			  So remove entries that you don't have. 
			  
			  Here, we just have rId1
  		 */
				
		// OK, now we can create a slide
		SlidePart slidePart = new SlidePart(new PartName("/ppt/slides/slide1.xml"));
		
		pp.addSlide(0, slidePart);
		
		// Slide layout part
		slidePart.addTargetPart(layoutPart);
		
		// Copy certain placeholders from layout part
		slidePart.setContents( extractFromLayout(layoutPart.getContents())  );		
		
				
		// Create and add shape
//		Shape sample = ((Shape)XmlUtils.unmarshalString(SAMPLE_SHAPE, Context.jcPML) );
//		slidePart.getContents().getCSld().getSpTree().getSpOrGrpSpOrGraphicFrame().add(sample);
		
		// All done: save it
		presentationMLPackage.save(new java.io.File(outputfilepath));

		System.out.println("\n\n done .. saved " + outputfilepath);
		
	}	
	
	private static Sld extractFromLayout(SldLayout sldLayout) {
		
		Sld sld = Context.getpmlObjectFactory().createSld();
		
		// Clone first
		sld.setCSld(XmlUtils.deepCopy(sldLayout.getCSld(), Context.jcPML));
		sld.setClrMapOvr(XmlUtils.deepCopy(sldLayout.getClrMapOvr(), Context.jcPML));
		
		// Then delete stuff
		sld.getCSld().setName(null);

		// Remove p:sp, if cNvPr name contains "Date Placeholder", "Footer Placeholder", "Slide Number Placeholder" 
		//       (and these are on the master?)
		List<Shape> deletions = new ArrayList<Shape>();
		for (Object o : sld.getCSld().getSpTree().getSpOrGrpSpOrGraphicFrame()) {
			
			//System.out.println(o.getClass().getName());
			if (o instanceof org.pptx4j.pml.Shape) {
				
				Shape shape = (Shape)o;
				if (shape.getNvSpPr()!=null
						&& shape.getNvSpPr().getCNvPr()!=null
						&& shape.getNvSpPr().getCNvPr().getName()!=null) {
					String name = shape.getNvSpPr().getCNvPr().getName();
					if (name.startsWith("Date Placeholder" )
							|| name.startsWith( "Footer Placeholder" )
							|| name.startsWith( "Slide Number Placeholder")) {
						
						deletions.add(shape);
					}
						
				}
			}
		}
		sld.getCSld().getSpTree().getSpOrGrpSpOrGraphicFrame().removeAll(deletions);
		
		// From remaining shapes ..
		for (Object o : sld.getCSld().getSpTree().getSpOrGrpSpOrGraphicFrame()) {
			if (o instanceof org.pptx4j.pml.Shape) {				
				Shape shape = (Shape)o;
				shape.setSpPr(new CTShapeProperties());
				if (shape.getTxBody()!=null) {
					
					shape.getTxBody().setLstStyle(null);
					
					for (CTTextParagraph p : shape.getTxBody().getP()) {
						p.getEGTextRun().clear();
					}
				}
			}
		}
		return sld;
	}
	
	
	private static String sldLayoutXML = "<p:sldLayout preserve=\"true\" type=\"title\" xmlns:p14=\"http://schemas.microsoft.com/office/powerpoint/2010/main\" xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\" xmlns:p=\"http://schemas.openxmlformats.org/presentationml/2006/main\">"
            + "<p:cSld name=\"Title Slide\">"
                + "<p:spTree>"
                    + "<p:nvGrpSpPr>"
                        + "<p:cNvPr id=\"1\" name=\"\"/>"
                        + "<p:cNvGrpSpPr/>"
                        + "<p:nvPr/>"
                    +"</p:nvGrpSpPr>"
                    + "<p:grpSpPr>"
                        + "<a:xfrm>"
                            + "<a:off x=\"0\" y=\"0\"/>"
                            + "<a:ext cx=\"0\" cy=\"0\"/>"
                            + "<a:chOff x=\"0\" y=\"0\"/>"
                            + "<a:chExt cx=\"0\" cy=\"0\"/>"
                        +"</a:xfrm>"
                    +"</p:grpSpPr>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"2\" name=\"Title 1\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph type=\"ctrTitle\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr>"
                            + "<a:xfrm>"
                                + "<a:off x=\"685800\" y=\"2130425\"/>"
                                + "<a:ext cx=\"7772400\" cy=\"1470025\"/>"
                            +"</a:xfrm>"
                        +"</p:spPr>"
                        + "<p:txBody>"
                            + "<a:bodyPr/>"
                            + "<a:lstStyle/>"
                            + "<a:p>"
                                + "<a:r>"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>Click to edit Master title style</a:t>"
                                +"</a:r>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"3\" name=\"Subtitle 2\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph idx=\"1\" type=\"subTitle\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr>"
                            + "<a:xfrm>"
                                + "<a:off x=\"1371600\" y=\"3886200\"/>"
                                + "<a:ext cx=\"6400800\" cy=\"1752600\"/>"
                            +"</a:xfrm>"
                        +"</p:spPr>"
                        + "<p:txBody>"
                            + "<a:bodyPr/>"
                            + "<a:lstStyle>"
                                + "<a:lvl1pPr algn=\"ctr\" indent=\"0\" marL=\"0\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl1pPr>"
                                + "<a:lvl2pPr algn=\"ctr\" indent=\"0\" marL=\"457200\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl2pPr>"
                                + "<a:lvl3pPr algn=\"ctr\" indent=\"0\" marL=\"914400\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl3pPr>"
                                + "<a:lvl4pPr algn=\"ctr\" indent=\"0\" marL=\"1371600\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl4pPr>"
                                + "<a:lvl5pPr algn=\"ctr\" indent=\"0\" marL=\"1828800\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl5pPr>"
                                + "<a:lvl6pPr algn=\"ctr\" indent=\"0\" marL=\"2286000\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl6pPr>"
                                + "<a:lvl7pPr algn=\"ctr\" indent=\"0\" marL=\"2743200\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl7pPr>"
                                + "<a:lvl8pPr algn=\"ctr\" indent=\"0\" marL=\"3200400\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl8pPr>"
                                + "<a:lvl9pPr algn=\"ctr\" indent=\"0\" marL=\"3657600\">"
                                    + "<a:buNone/>"
                                    + "<a:defRPr>"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl9pPr>"
                            +"</a:lstStyle>"
                            + "<a:p>"
                                + "<a:r>"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>Click to edit Master subtitle style</a:t>"
                                +"</a:r>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"4\" name=\"Date Placeholder 3\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph idx=\"10\" sz=\"half\" type=\"dt\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr/>"
                        + "<p:txBody>"
                            + "<a:bodyPr/>"
                            + "<a:lstStyle/>"
                            + "<a:p>"
                                + "<a:fld id=\"{A349BAA9-5664-4C19-AE26-9E57FBD43409}\" type=\"datetimeFigureOut\">"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>8/22/2016</a:t>"
                                +"</a:fld>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"5\" name=\"Footer Placeholder 4\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph idx=\"11\" sz=\"quarter\" type=\"ftr\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr/>"
                        + "<p:txBody>"
                            + "<a:bodyPr/>"
                            + "<a:lstStyle/>"
                            + "<a:p>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"6\" name=\"Slide Number Placeholder 5\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph idx=\"12\" sz=\"quarter\" type=\"sldNum\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr/>"
                        + "<p:txBody>"
                            + "<a:bodyPr/>"
                            + "<a:lstStyle/>"
                            + "<a:p>"
                                + "<a:fld id=\"{D0502544-975A-4C03-915C-805DB54C7530}\" type=\"slidenum\">"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>‹#›</a:t>"
                                +"</a:fld>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                +"</p:spTree>"
                + "<p:extLst>"
                    + "<p:ext uri=\"{BB962C8B-B14F-4D97-AF65-F5344CB8AC3E}\">"
                        + "<p14:creationId val=\"2664262704\"/>"
                    +"</p:ext>"
                +"</p:extLst>"
            +"</p:cSld>"
            + "<p:clrMapOvr>"
                + "<a:masterClrMapping/>"
            +"</p:clrMapOvr>"
        +"</p:sldLayout>";
	
	private static String slideMasterXML = "<p:sldMaster xmlns:p14=\"http://schemas.microsoft.com/office/powerpoint/2010/main\" xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\" xmlns:p=\"http://schemas.openxmlformats.org/presentationml/2006/main\" xmlns:r=\"http://schemas.openxmlformats.org/officeDocument/2006/relationships\">"
            + "<p:cSld>"
                + "<p:bg>"
                    + "<p:bgRef idx=\"1001\">"
                        + "<a:schemeClr val=\"bg1\"/>"
                    +"</p:bgRef>"
                +"</p:bg>"
                + "<p:spTree>"
                    + "<p:nvGrpSpPr>"
                        + "<p:cNvPr id=\"1\" name=\"\"/>"
                        + "<p:cNvGrpSpPr/>"
                        + "<p:nvPr/>"
                    +"</p:nvGrpSpPr>"
                    + "<p:grpSpPr>"
                        + "<a:xfrm>"
                            + "<a:off x=\"0\" y=\"0\"/>"
                            + "<a:ext cx=\"0\" cy=\"0\"/>"
                            + "<a:chOff x=\"0\" y=\"0\"/>"
                            + "<a:chExt cx=\"0\" cy=\"0\"/>"
                        +"</a:xfrm>"
                    +"</p:grpSpPr>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"2\" name=\"Title Placeholder 1\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph type=\"title\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr>"
                            + "<a:xfrm>"
                                + "<a:off x=\"457200\" y=\"274638\"/>"
                                + "<a:ext cx=\"8229600\" cy=\"1143000\"/>"
                            +"</a:xfrm>"
                            + "<a:prstGeom prst=\"rect\">"
                                + "<a:avLst/>"
                            +"</a:prstGeom>"
                        +"</p:spPr>"
                        + "<p:txBody>"
                            + "<a:bodyPr anchor=\"ctr\" bIns=\"45720\" lIns=\"91440\" rIns=\"91440\" rtlCol=\"false\" tIns=\"45720\" vert=\"horz\">"
                                + "<a:normAutofit/>"
                            +"</a:bodyPr>"
                            + "<a:lstStyle/>"
                            + "<a:p>"
                                + "<a:r>"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>Click to edit Master title style</a:t>"
                                +"</a:r>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"3\" name=\"Text Placeholder 2\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph idx=\"1\" type=\"body\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr>"
                            + "<a:xfrm>"
                                + "<a:off x=\"457200\" y=\"1600200\"/>"
                                + "<a:ext cx=\"8229600\" cy=\"4525963\"/>"
                            +"</a:xfrm>"
                            + "<a:prstGeom prst=\"rect\">"
                                + "<a:avLst/>"
                            +"</a:prstGeom>"
                        +"</p:spPr>"
                        + "<p:txBody>"
                            + "<a:bodyPr bIns=\"45720\" lIns=\"91440\" rIns=\"91440\" rtlCol=\"false\" tIns=\"45720\" vert=\"horz\">"
                                + "<a:normAutofit/>"
                            +"</a:bodyPr>"
                            + "<a:lstStyle/>"
                            + "<a:p>"
                                + "<a:pPr lvl=\"0\"/>"
                                + "<a:r>"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>Click to edit Master text styles</a:t>"
                                +"</a:r>"
                            +"</a:p>"
                            + "<a:p>"
                                + "<a:pPr lvl=\"1\"/>"
                                + "<a:r>"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>Second level</a:t>"
                                +"</a:r>"
                            +"</a:p>"
                            + "<a:p>"
                                + "<a:pPr lvl=\"2\"/>"
                                + "<a:r>"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>Third level</a:t>"
                                +"</a:r>"
                            +"</a:p>"
                            + "<a:p>"
                                + "<a:pPr lvl=\"3\"/>"
                                + "<a:r>"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>Fourth level</a:t>"
                                +"</a:r>"
                            +"</a:p>"
                            + "<a:p>"
                                + "<a:pPr lvl=\"4\"/>"
                                + "<a:r>"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>Fifth level</a:t>"
                                +"</a:r>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"4\" name=\"Date Placeholder 3\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph idx=\"2\" sz=\"half\" type=\"dt\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr>"
                            + "<a:xfrm>"
                                + "<a:off x=\"457200\" y=\"6356350\"/>"
                                + "<a:ext cx=\"2133600\" cy=\"365125\"/>"
                            +"</a:xfrm>"
                            + "<a:prstGeom prst=\"rect\">"
                                + "<a:avLst/>"
                            +"</a:prstGeom>"
                        +"</p:spPr>"
                        + "<p:txBody>"
                            + "<a:bodyPr anchor=\"ctr\" bIns=\"45720\" lIns=\"91440\" rIns=\"91440\" rtlCol=\"false\" tIns=\"45720\" vert=\"horz\"/>"
                            + "<a:lstStyle>"
                                + "<a:lvl1pPr algn=\"l\">"
                                    + "<a:defRPr sz=\"1200\">"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl1pPr>"
                            +"</a:lstStyle>"
                            + "<a:p>"
                                + "<a:fld id=\"{A349BAA9-5664-4C19-AE26-9E57FBD43409}\" type=\"datetimeFigureOut\">"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>8/22/2016</a:t>"
                                +"</a:fld>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"5\" name=\"Footer Placeholder 4\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph idx=\"3\" sz=\"quarter\" type=\"ftr\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr>"
                            + "<a:xfrm>"
                                + "<a:off x=\"3124200\" y=\"6356350\"/>"
                                + "<a:ext cx=\"2895600\" cy=\"365125\"/>"
                            +"</a:xfrm>"
                            + "<a:prstGeom prst=\"rect\">"
                                + "<a:avLst/>"
                            +"</a:prstGeom>"
                        +"</p:spPr>"
                        + "<p:txBody>"
                            + "<a:bodyPr anchor=\"ctr\" bIns=\"45720\" lIns=\"91440\" rIns=\"91440\" rtlCol=\"false\" tIns=\"45720\" vert=\"horz\"/>"
                            + "<a:lstStyle>"
                                + "<a:lvl1pPr algn=\"ctr\">"
                                    + "<a:defRPr sz=\"1200\">"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl1pPr>"
                            +"</a:lstStyle>"
                            + "<a:p>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                    + "<p:sp>"
                        + "<p:nvSpPr>"
                            + "<p:cNvPr id=\"6\" name=\"Slide Number Placeholder 5\"/>"
                            + "<p:cNvSpPr>"
                                + "<a:spLocks noGrp=\"true\"/>"
                            +"</p:cNvSpPr>"
                            + "<p:nvPr>"
                                + "<p:ph idx=\"4\" sz=\"quarter\" type=\"sldNum\"/>"
                            +"</p:nvPr>"
                        +"</p:nvSpPr>"
                        + "<p:spPr>"
                            + "<a:xfrm>"
                                + "<a:off x=\"6553200\" y=\"6356350\"/>"
                                + "<a:ext cx=\"2133600\" cy=\"365125\"/>"
                            +"</a:xfrm>"
                            + "<a:prstGeom prst=\"rect\">"
                                + "<a:avLst/>"
                            +"</a:prstGeom>"
                        +"</p:spPr>"
                        + "<p:txBody>"
                            + "<a:bodyPr anchor=\"ctr\" bIns=\"45720\" lIns=\"91440\" rIns=\"91440\" rtlCol=\"false\" tIns=\"45720\" vert=\"horz\"/>"
                            + "<a:lstStyle>"
                                + "<a:lvl1pPr algn=\"r\">"
                                    + "<a:defRPr sz=\"1200\">"
                                        + "<a:solidFill>"
                                            + "<a:schemeClr val=\"tx1\">"
                                                + "<a:tint val=\"75000\"/>"
                                            +"</a:schemeClr>"
                                        +"</a:solidFill>"
                                    +"</a:defRPr>"
                                +"</a:lvl1pPr>"
                            +"</a:lstStyle>"
                            + "<a:p>"
                                + "<a:fld id=\"{D0502544-975A-4C03-915C-805DB54C7530}\" type=\"slidenum\">"
                                    + "<a:rPr lang=\"en-US\" smtClean=\"false\"/>"
                                    + "<a:t>‹#›</a:t>"
                                +"</a:fld>"
                                + "<a:endParaRPr lang=\"en-US\"/>"
                            +"</a:p>"
                        +"</p:txBody>"
                    +"</p:sp>"
                +"</p:spTree>"
                + "<p:extLst>"
                    + "<p:ext uri=\"{BB962C8B-B14F-4D97-AF65-F5344CB8AC3E}\">"
                        + "<p14:creationId val=\"795255353\"/>"
                    +"</p:ext>"
                +"</p:extLst>"
            +"</p:cSld>"
            + "<p:clrMap accent1=\"accent1\" accent2=\"accent2\" accent3=\"accent3\" accent4=\"accent4\" accent5=\"accent5\" accent6=\"accent6\" bg1=\"lt1\" bg2=\"lt2\" folHlink=\"folHlink\" hlink=\"hlink\" tx1=\"dk1\" tx2=\"dk2\"/>"
            + "<p:sldLayoutIdLst>"
                + "<p:sldLayoutId id=\"2147483649\" r:id=\"rId1\"/>"
//                + "<p:sldLayoutId id=\"2147483650\" r:id=\"rId2\"/>"
//                + "<p:sldLayoutId id=\"2147483651\" r:id=\"rId3\"/>"
//                + "<p:sldLayoutId id=\"2147483652\" r:id=\"rId4\"/>"
//                + "<p:sldLayoutId id=\"2147483653\" r:id=\"rId5\"/>"
//                + "<p:sldLayoutId id=\"2147483654\" r:id=\"rId6\"/>"
//                + "<p:sldLayoutId id=\"2147483655\" r:id=\"rId7\"/>"
//                + "<p:sldLayoutId id=\"2147483656\" r:id=\"rId8\"/>"
//                + "<p:sldLayoutId id=\"2147483657\" r:id=\"rId9\"/>"
//                + "<p:sldLayoutId id=\"2147483658\" r:id=\"rId10\"/>"
//                + "<p:sldLayoutId id=\"2147483659\" r:id=\"rId11\"/>"
            +"</p:sldLayoutIdLst>"
            + "<p:txStyles>"
                + "<p:titleStyle>"
                    + "<a:lvl1pPr algn=\"ctr\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"0\"/>"
                        +"</a:spcBef>"
                        + "<a:buNone/>"
                        + "<a:defRPr kern=\"1200\" sz=\"4400\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mj-lt\"/>"
                            + "<a:ea typeface=\"+mj-ea\"/>"
                            + "<a:cs typeface=\"+mj-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl1pPr>"
                +"</p:titleStyle>"
                + "<p:bodyStyle>"
                    + "<a:lvl1pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-342900\" latinLnBrk=\"false\" marL=\"342900\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"•\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"3200\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl1pPr>"
                    + "<a:lvl2pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-285750\" latinLnBrk=\"false\" marL=\"742950\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"–\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"2800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl2pPr>"
                    + "<a:lvl3pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-228600\" latinLnBrk=\"false\" marL=\"1143000\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"•\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"2400\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl3pPr>"
                    + "<a:lvl4pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-228600\" latinLnBrk=\"false\" marL=\"1600200\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"–\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"2000\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl4pPr>"
                    + "<a:lvl5pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-228600\" latinLnBrk=\"false\" marL=\"2057400\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"»\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"2000\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl5pPr>"
                    + "<a:lvl6pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-228600\" latinLnBrk=\"false\" marL=\"2514600\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"•\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"2000\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl6pPr>"
                    + "<a:lvl7pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-228600\" latinLnBrk=\"false\" marL=\"2971800\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"•\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"2000\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl7pPr>"
                    + "<a:lvl8pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-228600\" latinLnBrk=\"false\" marL=\"3429000\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"•\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"2000\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl8pPr>"
                    + "<a:lvl9pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" indent=\"-228600\" latinLnBrk=\"false\" marL=\"3886200\" rtl=\"false\">"
                        + "<a:spcBef>"
                            + "<a:spcPct val=\"20000\"/>"
                        +"</a:spcBef>"
                        + "<a:buFont charset=\"0\" panose=\"020B0604020202020204\" pitchFamily=\"34\" typeface=\"Arial\"/>"
                        + "<a:buChar char=\"•\"/>"
                        + "<a:defRPr kern=\"1200\" sz=\"2000\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl9pPr>"
                +"</p:bodyStyle>"
                + "<p:otherStyle>"
                    + "<a:defPPr>"
                        + "<a:defRPr lang=\"en-US\"/>"
                    +"</a:defPPr>"
                    + "<a:lvl1pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"0\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl1pPr>"
                    + "<a:lvl2pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"457200\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl2pPr>"
                    + "<a:lvl3pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"914400\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl3pPr>"
                    + "<a:lvl4pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"1371600\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl4pPr>"
                    + "<a:lvl5pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"1828800\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl5pPr>"
                    + "<a:lvl6pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"2286000\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl6pPr>"
                    + "<a:lvl7pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"2743200\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl7pPr>"
                    + "<a:lvl8pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"3200400\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl8pPr>"
                    + "<a:lvl9pPr algn=\"l\" defTabSz=\"914400\" eaLnBrk=\"true\" hangingPunct=\"true\" latinLnBrk=\"false\" marL=\"3657600\" rtl=\"false\">"
                        + "<a:defRPr kern=\"1200\" sz=\"1800\">"
                            + "<a:solidFill>"
                                + "<a:schemeClr val=\"tx1\"/>"
                            +"</a:solidFill>"
                            + "<a:latin typeface=\"+mn-lt\"/>"
                            + "<a:ea typeface=\"+mn-ea\"/>"
                            + "<a:cs typeface=\"+mn-cs\"/>"
                        +"</a:defRPr>"
                    +"</a:lvl9pPr>"
                +"</p:otherStyle>"
            +"</p:txStyles>"
        +"</p:sldMaster>";

	
	private static String SAMPLE_SHAPE = 			
		"<p:sp   xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\" xmlns:r=\"http://schemas.openxmlformats.org/officeDocument/2006/relationships\" xmlns:p=\"http://schemas.openxmlformats.org/presentationml/2006/main\">"
		+ "<p:nvSpPr>"
		+ "<p:cNvPr id=\"4\" name=\"Title 3\" />"
		+ "<p:cNvSpPr>"
			+ "<a:spLocks noGrp=\"1\" />"
		+ "</p:cNvSpPr>"
		+ "<p:nvPr>"
			+ "<p:ph type=\"title\" />"
		+ "</p:nvPr>"
	+ "</p:nvSpPr>"
	+ "<p:spPr />"
	+ "<p:txBody>"
		+ "<a:bodyPr />"
		+ "<a:lstStyle />"
		+ "<a:p>"
			+ "<a:r>"
				+ "<a:rPr lang=\"en-US\" smtClean=\"0\" />"
				+ "<a:t>Hello World</a:t>"
			+ "</a:r>"
			+ "<a:endParaRPr lang=\"en-US\" />"
		+ "</a:p>"
	+ "</p:txBody>"
+ "</p:sp>";

	
}
